"""
renewal.py

Implementation of the Renewal operator F for the tick‑flip algebra.

The renewal operator moves mass one layer outward on the context
lattice, subject to a hard boundary at r = ±N.  In terms of the
distribution array this corresponds to shifting all entries one index
towards the start of the array.  Mass initially at the outermost layer
(index 0) remains there and collects mass from its neighbour.  No mass
appears beyond the right‑hand boundary.
"""

import numpy as np
from .tick_state import TickState


def renewal(state: TickState) -> TickState:
    """Apply one renewal (outward flip) step to a tick state.

    Parameters
    ----------
    state : TickState
        The input tick state.  Its distribution is not mutated.

    Returns
    -------
    TickState
        A new tick state with mass shifted one layer outward.  At the
        left boundary the existing mass and the mass shifting in from the
        next layer are added together.  Mass that would fall off the
        right boundary is discarded (i.e. the rightmost entry of the
        output distribution is set to zero).
    """
    dist = state.distribution
    L = dist.size
    # create a new array to avoid mutating the input distribution
    new = np.zeros_like(dist)
    if L == 0:
        return TickState(new, state.N)
    # At the left boundary (index 0) collect mass from itself and from index 1
    if L > 1:
        new[0] = dist[0] + dist[1]
    else:
        new[0] = dist[0]
    # Shift interior values one step towards the left: new[i] = dist[i+1]
    if L > 2:
        new[1 : L - 1] = dist[2:]
    # Rightmost entry loses mass since nothing shifts into it
    new[L - 1] = 0.0
    return TickState(new, state.N)


def F(state: TickState) -> TickState:
    """
    Alias for the renewal operator.

    Parameters
    ----------
    state : TickState
        The input tick state.

    Returns
    -------
    TickState
        The result of applying the renewal operator to ``state``.
    """
    return renewal(state)
